function fault_slip_patch_bounds_check
%
% Produces figures for graphical check of finite difference and Chebyshev results.
% Can be run stand-alone or through fault_slip_main. 
%

clear vars
close all

% ---------------------------------------------------------------------------------------------
% Set generic input parameters (plot-specific ones will be set later):
% ---------------------------------------------------------------------------------------------
a = 75; % see Figure 1, m
alpha = 0.9; % Biot's coefficient, -
b = 150; % see Figure 1, m
delta_c = 0.02; % critical slip distance for slip weakening 0, m
D_center = 3500; % reservoir depth @ y = 0, m
g = 9.81; % acceleration of gravity, m/s^2
G = 6500e6; % shear modulus, Pa
kappa = 0; % cohesion, -
K_eff = 0.5; % initial effective vertical/horizontal stress ratio, -
mu_dyn = 0.20; % dynamic friction coefficient 0, -
mu_st = 0.52; % static friction coefficient, -
nu = 0.15; % Poisson's ratio, -
p_0_center = 35e6; % initial pressure @ y = 0, Pa
phi = 0.15; % porosity, -
rho_f = 1020; % fluid density, kg/m^3
rho_s = 2650; % solid density, kg/m^3
sigma_v_0 = 0; % initial vertical stress @ y = 0, Pa. If set equal to zero, the stress is ...
    % computed from the weigth of the overburden; see function fault_slip_sigma.m.
theta = 70*pi/180; % dip angle, rad

% ---------------------------------------------------------------------------------------------
% Set program flow flags and computational parameters:
% ---------------------------------------------------------------------------------------------
f_damp = 0.5; % damping factor for iterative update of slip-weakening and coupled ...
    % simulation (only relevant for flag_cheb == 1)
flag_check = 1; % flag to trigger graphical check of finite difference and Chebyshev results
    % 0 = check disabled (default) 
    % 1 = check enabled (for diagnostics; causes overwriting of all input from main file; ...
    %     see fault_slip_patch_bounds_check.m)
flag_cont = 1; % flag to force continuation in case of non-convergence of slip-weakening ...
    % and coupled simulation (only relevant for flag_cheb == 1)
    % 0 = continuation disabled (for diagnostics) 
    % 1 = continuation enabled (recommended for regular use)
flag_cor = 1; % flag to force correct sign on slip gradient (only relevant for flag_cheb = 0)
    % 0 = forcing switched off (for diagnostics)
    % 1 = forcing switched on (recommended for regular use) 
flag_decouple = 0; % flag to force decoupling between slip patches (only relevant for ...
    % flag_fric > 1; for flag_fric = 1, coupling is always active.)
    % 0 = default coupled simulation; correct but takes more time; not always necessary 
    % 1 = decoupled simulation; faster but possibly inaccurate (only available for ...
    %     slip-weakening)
flag_echo = 1; % flag to trigger writing to screen of simulation progress
    % 0 = no echo
    % 1 = limited echo; only info about main simulation components
    % 2 = full echo; info about each simulation step
flag_eig_use = 2; % flag to determine use of eigenvalues (only relevant for flag_fric > 1)
    % 0 = used to check accuracy of simulated slip length (Delta y*_sim ~= Delta y*_eig?) 
    %     (for diagnostics)
    % 1 = used for seismicity detection (stops simulation when Delta y*_sim > Delta y*_eig)
    %     (for diagnostics)
    % 2 = alternative: numerical derivative used for seismicity detection (stops simulation 
    %     when (dp/dDelta y) drops below tol_deriv); requires flag_lag = 0  
    %     (recommended for regular use)
tol_deriv = 1e-5; % tolerance for scaled numerical derivative (dp/dDelta y)*(Delta_y/A), ...
    % where A = G / (2*pi*(1-nu)), - 
flag_lag = 0; % flag to trigger lagging of eigen problem computation with one presure step
    % (only relevant for flag_fric > 1)
    % 0 = lagging switched off (recommended for regular use)
    % 1 = lagging switched on (for diagnostics) 
flag_M = 0; % flag to compute seismic moment (can be time consuming)
    % 0 = computation switched off
    % 1 = computation switched on
flag_paper = 0; % flag to trigger plotting of figure for paper
    % 0 = default plots
    % 1 = plot for paper; causes overwriting of all input from main file; see ...
    %     fault_slip_patch_bounds_all.m
flag_reg = 1; % flag to control regularization of slip-dependent friction coefficient and ...
    % Coulomb stresses:
    % 0 = regularization switched off (for diagnostics)
    % 1 = regularization switched on (recommended for regular use)
eps_reg = 0; % regularization parameter for slip-dependent friction coefficient, m
% eps_reg = 2e-4; % regularization parameter for slip-dependent friction coefficient, m
% eps_reg = 2e-3; % regularization parameter for slip-dependent friction coefficient, m
eta_reg = 0.10; % regularization parameter for Coulomb stresses, m
% eta_reg = 0.20; % regularization parameter for Coulomb stresses, m
% eta_reg = 0.50; % regularization parameter for Coulomb stresses, m
flag_top_only = 1; % flag to force occurrence of seismicity in top patch only (only ...
    % relevant for flag_cheb == 1)  
    % 0 = forcing switched off (for diagnostics)
    % 1 = forcing switched on (recommended for regular use)

% Computational parameters for patch size computation:
n_max_int = 2000; % max. allowed total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
n_min_int = 100;  % min. required total nr. of integr. pnts, -. Only relevant for flag_cheb = 0
cheb_terms = 250; % number of terms in Chebyshev polynomial approximation for semi-...
    % analytical integration.
max_iter_coupled = 100; % maximum allowed number of iterations for patch size computation

% Computational parameters for pressure stepping:
p_max_bounds = -10e6; % highest incremental pressure (negative for production), Pa
p_min_bounds = -30e6; % lowest incremental pressure (negative for production), Pa
Delta_p_init = -0.5e6; % initial pressure step size (negative for production), Pa
Delta_p_max = -1e6; % maximum allowed pressure step size (negative for production), Pa
small_patch = 0.05; % minimum patch size to trigger slip computation during pr. stepping, m

% ---------------------------------------------------------------------------------------------
% Set plotting flags and parameters:
% ---------------------------------------------------------------------------------------------
% Plotting flags:
flag_plot_cap = 1; % flag for plot captions (black) and plot file location captions (red) 
    % 0 = no captions
    % 1 = with captions
flag_plot_diag = 0; % flag for diagnostic plots:
    % 0 = plotting switched off
    % 1 = plotting switched on
flag_plot_eig = 0; % flag for eigenvalue plot (only relevant for slip-weakening friction):
    % 0 = plotting switched off
    % 1 = plotting switched on
flag_plot_fields = 0; % flag for field plots (slip, slip gradient etc. as function of p and y):
    % 0 = plotting switched off
    % 1 = plotting switched on
flag_plot_snap = 0; % flag for snapshot plots:
    % 0 = plotting switched off
    % 1 = plotting switched on   
% Parameters for snapshot plots of stresses, slip etc.:
    % Note 1: all comments refer to parameter settings for example 1 from the paper.
    % Note 2: different plots are created for zero, constant or slip-weakening ...
    % friction. Moreover for p_inc_02, additional plots are created in case of ...
    % slip-weakening friction. See fault_slip_patch_bounds.m for further info.  
p_inc_01 = 0; % dummy
p_inc_02 = 0;
p_inc_03 = 0;
p_inc_04 = 0;
p_inc_05 = 0;
y_plot_max_delta_1 =  100; % covers both slip patches (in detail), m
y_plot_min_delta_1 = -100; % covers both slip patches (in detail), m
y_plot_max_delta_2 =   80; % covers top patch only (further detail), m
y_plot_min_delta_2 =   50; % covers top patch only (further detail), m
y_plot_max_sigma   =  250; % covers both slip patches (overview), m
y_plot_min_sigma   = -250; % covers both slip patches (overview), m
y_plot_n_pt = 2000; % number of points between y_plot_min_... and y_plot_max_... 

% Parameters for plots of Coulomb stress zeros and slip patch boundaries as a function of ...
%   depletion pressure:
y_plot_max_bounds =  100; % covers both slip patches (in detail), m
y_plot_min_bounds = -100; % covers both slip patches (in detail), m

% Colors for line plots:
blue = [0/255 128/255 255/255];
gray = [235/255 235/255 235/255];
green = [0/255 204/255 0/255];
orange = [255/255 185/255 0/255];
colors = [blue; gray; green; orange];

% ---------------------------------------------------------------------------------------------
% End of generic input. Start of plot-specific input, simulation and plotting.
% ---------------------------------------------------------------------------------------------
flag_fric = 1; % constant friction
Delta_y_int = 0.10; % spatial integration step size, m. Only relevant for flag_cheb = 0
tol_rel_coupled = 1e-4; % relative tolerance for for patch size computation
Delta_p_min = -0.10e6; % minimum allowed pressure step size (negative for production), Pa
Delta_y_target = 0.50; % target change in y value during pressure stepping, m

% Create first part of plot:
fprintf('\rPart 1: Chebyshev integration. Constant friction.\r');
flag_cheb = 1; % semi-analytical integration with Chebyshev polynomials
[~,~,~,~,~,p_plot,p_2_plot,p_3_plot,~,p_tilde_2_plot,p_tilde_3_plot,...
    p_tilde_unc_2_plot,p_tilde_unc_3_plot,y_1_plot,y_2_plot,y_3_plot,y_4_plot,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,...
    y_tilde_unc_2_plot,y_tilde_unc_3_plot] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,g,G,kappa,...
    K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);

figure % Slip patch boundaries vs. p_inc
hold on
plot(-p_plot/1e6,y_1_plot,'-','Color',green,'LineWidth',1.5);
plot(-p_2_plot/1e6,y_2_plot,'-','Color',green,'LineWidth',1.5);
plot(-p_3_plot/1e6,y_3_plot,'-','Color',green,'LineWidth',1.5);
plot(-p_plot/1e6,y_4_plot,'-','Color',green,'LineWidth',1.5);
plot(-p_tilde_unc_2_plot/1e6,y_tilde_unc_2_plot,':','Color',blue,'LineWidth',1.5)
plot(-p_tilde_unc_3_plot/1e6,y_tilde_unc_3_plot,':','Color',blue,'LineWidth',1.5)
plot(-p_plot/1e6,y_tilde_1_plot,'-','Color',blue,'LineWidth',1.5)
plot(-p_tilde_2_plot/1e6,y_tilde_2_plot,'-','Color',blue,'LineWidth',1.5)
plot(-p_tilde_3_plot/1e6,y_tilde_3_plot,'-','Color',blue,'LineWidth',1.5)
plot(-p_plot/1e6,y_tilde_4_plot,'-','Color',blue,'LineWidth',1.5)

% Create second part of plot:
fprintf('Part 2: Numerical integration. Constant friction.\r');
flag_cheb = 0; % numerical integration with near-uniform staggered grids for y and xi
[~,~,~,~,~,p_plot,p_2_plot,p_3_plot,~,p_tilde_2_plot,p_tilde_3_plot,...
    p_tilde_unc_2_plot,p_tilde_unc_3_plot,y_1_plot,y_2_plot,y_3_plot,y_4_plot,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,...
    y_tilde_unc_2_plot,y_tilde_unc_3_plot] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_cont,flag_cor,flag_check,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,g,G,kappa,...
    K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);
plot(-p_plot/1e6,y_1_plot,'--','Color','k','LineWidth',1.5);
plot(-p_2_plot/1e6,y_2_plot,'--','Color','k','LineWidth',1.5);
plot(-p_3_plot/1e6,y_3_plot,'--','Color','k','LineWidth',1.5);
plot(-p_plot/1e6,y_4_plot,'--','Color','k','LineWidth',1.5);
plot(-p_tilde_unc_2_plot/1e6,y_tilde_unc_2_plot,'--','Color','k','LineWidth',1.5)
plot(-p_tilde_unc_3_plot/1e6,y_tilde_unc_3_plot,'--','Color','k','LineWidth',1.5)
plot(-p_plot/1e6,y_tilde_1_plot,'--','Color','k','LineWidth',1.5)
plot(-p_tilde_2_plot/1e6,y_tilde_2_plot,'--','Color','k','LineWidth',1.5)
plot(-p_tilde_3_plot/1e6,y_tilde_3_plot,'--','Color','k','LineWidth',1.5)
plot(-p_plot/1e6,y_tilde_4_plot,'--','Color','k','LineWidth',1.5)
line([ 0 35],[   0    0],'LineStyle','-','Color','k','LineWidth',0.5) % horizontal line
line([ 0 35],[   a    a],'LineStyle',':','Color','k','LineWidth',0.5)
line([ 0 35],[  -a   -a],'LineStyle',':','Color','k','LineWidth',0.5)
line([25 25],[y_plot_min_bounds y_plot_max_bounds],'LineStyle','--','Color','k',...
    'LineWidth',0.5)
text(30.3,-75,'$\it y_1$','Interpreter','latex','FontSize',11)
text(28.9,-20,'$\it y_2$','Interpreter','latex','FontSize',11)
text(28.8, 20,'$\it y_3$','Interpreter','latex','FontSize',11)
text(30.3, 75,'$\it y_4$','Interpreter','latex','FontSize',11)
text(30.3,-88,'$\it \tilde{y}_1$','Interpreter','latex','FontSize',11)
text(26.8,-20,'$\it \tilde{y}_2$','Interpreter','latex','FontSize',11)
text(26.7, 20,'$\it \tilde{y}_3$','Interpreter','latex','FontSize',11)
text(30.3, 88,'$\it \tilde{y}_4$','Interpreter','latex','FontSize',11)
    xlabel('$\it -p$ (MPa)','Interpreter','latex','FontSize',11)
    ylabel('$\it y_i, \tilde{y}_i$ (m)','Interpreter','latex','FontSize',11)
    axis([-p_max_bounds/1e6 -p_min_bounds/1e6 y_plot_min_bounds y_plot_max_bounds])
    box on

% Create third part of plot:
fprintf('Part 3: Chebyshev integration. Slip weakening friction with mu_dyn = 0.20.\r');
flag_fric = 2; % slip-weakening friction
Delta_y_int = 0.01; % spatial integration step size, m. Only relevant for flag_cheb = 0
tol_rel_coupled = 1e-7; % relative tolerance for for patch size computation
Delta_p_min = -0.005e6; % minimum allowed pressure step size (negative for production), Pa
Delta_y_target = 0.10; % target change in y value during pressure stepping, m
flag_cheb = 1; % numerical integration with near-uniform staggered grids for y and xi
[~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,...
    g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);
plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color','r','LineWidth',1.5)
plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color','r',...
    'LineWidth',1.5)
plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color','r',...
    'LineWidth',1.5)
plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color','r','LineWidth',1.5)
line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
    'LineStyle',':','Color','r','LineWidth',1.0)
text(-p_seismic/1e6+0.3,max(y_tilde_1_plot(end-1),y_plot_min_bounds)-6,...
    '$\it \tilde{y}_1$','Interpreter','latex','FontSize',11)
text(-p_seismic/1e6+0.3,y_tilde_2_plot(end-1)+7,'$\it \tilde{y}_2$','Interpreter',...
    'latex','FontSize',11)
text(-p_seismic/1e6+0.3,y_tilde_3_plot(end-1)-5,'$\it \tilde{y}_3$','Interpreter',...
    'latex','FontSize',11)
text(-p_seismic/1e6+0.3,min(y_tilde_4_plot(end-1),y_plot_max_bounds)+7,...
    '$\it \tilde{y}_4$','Interpreter','latex','FontSize',11)

% Create fourth part of plot:
fprintf('Part 4: Numerical integration. Slip weakening friction with mu_dyn = 0.20.\r');
flag_cheb = 0; % semi-analytical integration with Chebyshev polynomials
[~,~,~,~,~,p_plot,~,~,~,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,...
    g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);
plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'--','Color','k','LineWidth',1.5)
plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'--','Color','k',...
    'LineWidth',1.5)
plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'--','Color','k',...
    'LineWidth',1.5)
plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'--','Color','k','LineWidth',1.5)

% Create fifth part of plot (repeat for higher mu_dyn):
fprintf('Part 5: Chebyshev integration. Slip weakening friction with mu_dyn = 0.40.\r');
% flag_fric = 2; % slip-weakening friction
% Delta_y_int = 0.01; % spatial integration step size, m. Only relevant for flag_cheb = 0
% tol_rel_coupled = 1e-7; % relative tolerance for for patch size computation
% Delta_p_min = -0.005e6; % minimum allowed pressure step size (negative for production), Pa
% Delta_y_target = 0.10; % target change in y value during pressure stepping, m
mu_dyn = 0.40; % dynamic friction coefficient, -
flag_cheb = 1; % semi-analytical inversion through integration with Chebyshev polynomials 
flag_eig_use = 1; % used for seismicity detection
flag_lag = 1; % lagging switched on !!! 
[~,~,~,~,~,p_plot,~,~,p_seismic,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,...
    g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);
plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'-','Color',orange,'LineWidth',1.5)
plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'-','Color',orange,...
    'LineWidth',1.5)
plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'-','Color',orange,...
    'LineWidth',1.5)
plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'-','Color',orange,'LineWidth',1.5)
line([-p_seismic/1e6 -p_seismic/1e6],[y_plot_min_bounds y_plot_max_bounds],...
    'LineStyle',':','Color',orange,'LineWidth',1.0)
text(-p_seismic/1e6+0.3,max(y_tilde_1_plot(end-1),y_plot_min_bounds)-6,...
    '$\it \tilde{y}_1$','Interpreter','latex','FontSize',11)
text(-p_seismic/1e6+0.3,y_tilde_2_plot(end-1)+7,'$\it \tilde{y}_2$','Interpreter',...
    'latex','FontSize',11)
text(-p_seismic/1e6+0.3,y_tilde_3_plot(end-1)-5,'$\it \tilde{y}_3$','Interpreter',...
    'latex','FontSize',11)
text(-p_seismic/1e6+0.3,min(y_tilde_4_plot(end-1),y_plot_max_bounds)+7,...
    '$\it \tilde{y}_4$','Interpreter','latex','FontSize',11)

% Create sixth part of plot:
fprintf('Part 6: Numerical integration. Slip weakening friction with mu_dyn = 0.40.\r');
flag_cheb = 0; 
[~,~,~,~,~,p_plot,~,~,~,p_tilde_2_plot,p_tilde_3_plot,~,~,~,~,~,~,...
    y_tilde_1_plot,y_tilde_2_plot,y_tilde_3_plot,y_tilde_4_plot,~,~] = ...
    fault_slip_patch_bounds(a,alpha,b,cheb_terms,colors,delta_c,...
    D_center,Delta_p_init,Delta_p_max,Delta_p_min,Delta_y_int,Delta_y_target,eps_reg,...
    eta_reg,f_damp,flag_cheb,flag_check,flag_cont,flag_cor,flag_decouple,flag_echo,...
    flag_eig_use,flag_fric,flag_lag,flag_M,flag_paper,flag_plot_cap,flag_plot_diag,...
    flag_plot_eig,flag_plot_fields,flag_plot_snap,flag_reg,flag_top_only,...
    g,G,kappa,K_eff,max_iter_coupled,mu_dyn,mu_st,n_max_int,n_min_int,nu,p_0_center,...
    p_inc_01,p_inc_02,p_inc_03,p_inc_04,p_inc_05,...
    p_max_bounds,p_min_bounds,...
    phi,rho_f,rho_s,sigma_v_0,small_patch,theta,tol_deriv,tol_rel_coupled,...
    y_plot_max_bounds,y_plot_max_delta_1,y_plot_max_delta_2,y_plot_max_sigma,...
    y_plot_min_bounds,y_plot_min_delta_1,y_plot_min_delta_2,y_plot_min_sigma,...
    y_plot_n_pt);
plot(-p_plot(1:end-1)/1e6,y_tilde_1_plot(1:end-1),'--','Color','k','LineWidth',1.5)
plot(-p_tilde_2_plot(1:end-1)/1e6,y_tilde_2_plot(1:end-1),'--','Color','k',...
    'LineWidth',1.5)
plot(-p_tilde_3_plot(1:end-1)/1e6,y_tilde_3_plot(1:end-1),'--','Color','k',...
    'LineWidth',1.5)
plot(-p_plot(1:end-1)/1e6,y_tilde_4_plot(1:end-1),'--','Color','k','LineWidth',1.5)

fprintf('See plot for comparison. Chebyshev integration: solid coloured lines; Numerical integration: dashed black lines. \r');
